// LinkedIn Lead Extractor - Send to Web Server
// Replaces Google Sheets with direct server communication

console.log('🚀 LinkedIn Lead Extractor Background - Server Mode');

// Server configuration (set via popup or storage)
let SERVER_URL = "https://100milliondollarbag.com/";
let API_KEY = null; // Optional API key for authentication

// Load server configuration
async function loadServerConfig() {
  try {
    const result = await chrome.storage.sync.get(['serverUrl', 'apiKey']);
    SERVER_URL = result.serverUrl || null;
    API_KEY = result.apiKey || null;
    
    if (SERVER_URL) {
      // Ensure URL ends with /
      if (!SERVER_URL.endsWith('/')) {
        SERVER_URL += '/';
      }
      console.log('✅ Server URL loaded:', SERVER_URL);
    } else {
      console.warn('⚠️ Server URL not configured');
    }
  } catch (error) {
    console.error('❌ Failed to load server config:', error);
  }
}

// Initialize on startup
loadServerConfig();

// Listen for config updates
chrome.storage.onChanged.addListener((changes, areaName) => {
  if (areaName === 'sync' && (changes.serverUrl || changes.apiKey)) {
    loadServerConfig();
  }
});

// Send leads to server
async function sendToServer(payload) {
  try {
    if (!SERVER_URL) {
      throw new Error('Server URL not configured. Please set it in the extension popup.');
    }
    
    console.log(`📤 Sending ${payload.leads.length} leads to server: ${SERVER_URL}`);
    
    if (!payload.leads || payload.leads.length === 0) {
      return { success: false, error: 'No leads to send', insertedCount: 0 };
    }
    
    // Prepare leads data
    const leadsData = payload.leads.map(lead => ({
      leadId: lead.leadId || '',
      fullName: String(lead.fullName || '').trim(),
      company: String(lead.company || '').trim(),
      state: String(lead.state || '').trim(),
      country: String(lead.country || '').trim(),
      extractedAt: lead.extractedAt || new Date().toISOString(),
      workerId: String(lead.workerId || payload.workerID || 'Unknown').trim()
    })).filter(lead => lead.fullName && lead.company); // Filter invalid leads
    
    if (leadsData.length === 0) {
      return { success: false, error: 'No valid leads to send after validation', insertedCount: 0 };
    }
    
    // Send to server
    const headers = {
      'Content-Type': 'application/json'
    };
    
    if (API_KEY) {
      headers['X-API-Key'] = API_KEY;
    }
    
    const response = await fetch(`${SERVER_URL}api/leads/upload`, {
      method: 'POST',
      headers: headers,
      body: JSON.stringify({
        leads: leadsData,
        workerId: payload.workerID || 'Unknown'
      })
    });
    
    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Server error: ${response.status} - ${errorText}`);
    }
    
    const result = await response.json();
    
    if (result.success) {
      console.log(`✅ Successfully sent ${result.insertedCount || leadsData.length} leads to server`);
      return {
        success: true,
        message: `Added ${result.insertedCount || leadsData.length} leads`,
        insertedCount: result.insertedCount || leadsData.length,
        duplicates: result.duplicates || 0
      };
    } else {
      throw new Error(result.error || 'Server returned error');
    }
  } catch (error) {
    console.error('❌ Send to server failed:', error);
    return { success: false, error: error.message, insertedCount: 0 };
  }
}

// Check for duplicates on server
async function checkDuplicates(leads) {
  try {
    if (!SERVER_URL) {
      // If server not configured, allow all (no duplicate check)
      return { success: true, duplicates: [] };
    }
    
    const headers = {
      'Content-Type': 'application/json'
    };
    
    if (API_KEY) {
      headers['X-API-Key'] = API_KEY;
    }
    
    const response = await fetch(`${SERVER_URL}api/leads/check-duplicates`, {
      method: 'POST',
      headers: headers,
      body: JSON.stringify({ leads })
    });
    
    if (!response.ok) {
      // If check fails, allow all (fail open)
      return { success: true, duplicates: [] };
    }
    
    const result = await response.json();
    return {
      success: true,
      duplicates: result.duplicates || []
    };
  } catch (error) {
    console.error('❌ Duplicate check failed:', error);
    // Fail open - allow all if check fails
    return { success: true, duplicates: [] };
  }
}

// Get stats from server
async function getMyStats(workerID) {
  try {
    if (!SERVER_URL || !workerID) {
      return { success: false, error: 'Server URL or Worker ID not configured' };
    }
    
    const headers = {};
    if (API_KEY) {
      headers['X-API-Key'] = API_KEY;
    }
    
    const response = await fetch(`${SERVER_URL}api/leads/stats?workerId=${encodeURIComponent(workerID)}`, {
      method: 'GET',
      headers: headers
    });
    
    if (!response.ok) {
      return { success: false, error: `Server error: ${response.status}` };
    }
    
    const result = await response.json();
    return {
      success: true,
      daily: result.daily || 0,
      weekly: result.weekly || 0,
      monthly: result.monthly || 0,
      total: result.total || 0
    };
  } catch (error) {
    console.error('❌ Get stats failed:', error);
    return { success: false, error: error.message };
  }
}

// Download user's data from server
async function downloadMyData(workerID) {
  try {
    if (!SERVER_URL || !workerID) {
      return { success: false, error: 'Server URL or Worker ID not configured' };
    }
    
    const headers = {};
    if (API_KEY) {
      headers['X-API-Key'] = API_KEY;
    }
    
    const response = await fetch(`${SERVER_URL}api/leads/download?workerId=${encodeURIComponent(workerID)}`, {
      method: 'GET',
      headers: headers
    });
    
    if (!response.ok) {
      return { success: false, error: `Server error: ${response.status}` };
    }
    
    const result = await response.json();
    return {
      success: true,
      data: result.data || []
    };
  } catch (error) {
    console.error('❌ Download failed:', error);
    return { success: false, error: error.message };
  }
}

// Message handler
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  console.log('📨 Message received:', request.action);
  
  let responseSent = false;
  
  const safeSendResponse = (data) => {
    if (!responseSent) {
      try {
        sendResponse(data);
        responseSent = true;
      } catch (error) {
        console.warn('⚠️ Could not send response:', error.message);
      }
    }
  };
  
  if (request.action === 'sendToServer') {
    sendToServer(request.payload)
      .then(response => {
        safeSendResponse(response);
      })
      .catch(error => {
        safeSendResponse({ success: false, error: error.message });
      });
    return true; // Async response
  }
  
  if (request.action === 'checkDuplicates') {
    checkDuplicates(request.leads)
      .then(response => {
        safeSendResponse(response);
      })
      .catch(error => {
        safeSendResponse({ success: false, error: error.message, duplicates: [] });
      });
    return true;
  }
  
  if (request.action === 'getMyStats') {
    getMyStats(request.workerID)
      .then(response => {
        safeSendResponse(response);
      })
      .catch(error => {
        safeSendResponse({ success: false, error: error.message });
      });
    return true;
  }
  
  if (request.action === 'downloadMyData') {
    downloadMyData(request.workerID)
      .then(response => {
        safeSendResponse(response);
      })
      .catch(error => {
        safeSendResponse({ success: false, error: error.message });
      });
    return true;
  }
  
  if (request.action === 'getServerConfig') {
    safeSendResponse({ serverUrl: SERVER_URL, apiKey: API_KEY });
    return false; // Sync response
  }
  
  if (request.action === 'setServerConfig') {
    chrome.storage.sync.set({
      serverUrl: request.serverUrl,
      apiKey: request.apiKey
    }).then(() => {
      loadServerConfig();
      safeSendResponse({ success: true });
    });
    return true;
  }
  
  return false;
});

console.log('✅ Background script ready - Server mode');

